<?php

namespace App\Http\Controllers\Dashboard;

use App\Models\WithdrawHistory;
use App\Http\Controllers\Controller;
use App\Models\AppData;
use App\Notifications\WithdrawRequestAcceptNotification;
use App\Notifications\WithdrawRequestRejectNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;

class WithdrawHistoryController extends Controller
{
    public function index(Request $request)
    {
        if ($request->has('searchValue')) {
            $searchValue = $request->searchValue;
            $withdrawHistories = WithdrawHistory::with('user')
                ->whereHas('user', function ($query) use ($searchValue) {
                    $query->where('name', 'LIKE', '%' . $searchValue . '%')
                        ->where('role', 'user')
                        ->orWhere('phone', 'LIKE', '%' . $searchValue . '%');
                })->latest()->paginate(250);
            return view('dashboard.withdraw-history.index', compact('withdrawHistories', 'searchValue'));
        }
        $withdrawHistories  = WithdrawHistory::with('user')
            ->whereHas('user', function ($query) {
                $query->where('role', 'user');
            })->latest()->paginate(25);
        return view('dashboard.withdraw-history.index', compact('withdrawHistories'));
    }

    public function agent(Request $request)
    {
        if ($request->has('searchValue')) {
            $searchValue = $request->searchValue;
            $withdrawHistories = WithdrawHistory::with('user')
                ->whereHas('user', function ($query) use ($searchValue) {
                    $query->where('name', 'LIKE', '%' . $searchValue . '%')
                        ->orWhere('phone', 'LIKE', '%' . $searchValue . '%')
                        ->where('role', 'agent');
                })->latest()->paginate(250);
            return view('dashboard.withdraw-history.agent', compact('withdrawHistories', 'searchValue'));
        }
        $withdrawHistories  = WithdrawHistory::with('user')
            ->whereHas('user', function ($query) {
                $query->where('role', 'agent');
            })->latest()->paginate(25);
        return view('dashboard.withdraw-history.agent', compact('withdrawHistories'));
    }

    //acceptallapi
    public function acceptAllRequestApi()
    {
        if (!env('ENABLE_PAYOUT')) return back()->with('success', 'Payout is disabled');
        if (env('PAYOUT_API_NAME') !== 'ibr_pay') return back()->with('success', 'BULK Payout API can work only with IBR Pay, Please Contact Support');
        $withdraws = WithdrawHistory::where('status', 'pending')
            ->with('user', 'user.withdrawDetails')
            ->get();
        foreach ($withdraws as $withdraw) {
            $user = $withdraw->user;
            $client = new Client();
            $url = 'https://ibrpay.com/api/PayoutLive.aspx';
            $data = [
                'APIID' => env('GATEWAY_IBR_PAY_APIID'),
                'Token' => env('GATEWAY_IBR_PAY_API_TOKEN'),
                'MethodName' => 'payout',
                'OrderID' => Str::random(12),
                'Name' => $user->withdrawDetails->account_holder_name,
                'Amount' => $withdraw->amount,
                'number' => $user->withdrawDetails->account_number,
                'ifsc' => $user->withdrawDetails->account_ifsc_code,
                'PaymentType' => 'personal',
                'CustomerMobileNo' => $user->phone,
            ];

            $success_request_count = 0;
            $failed_request_count = 0;
            $pending_request_count = 0;
            $status_failed_request_count = 0;
            $api_failed_request_count = 0;

            try {
                $response = $client->post($url, [
                    'json' => $data
                ]);

                $statusCode = $response->getStatusCode();
                $body = $response->getBody()->getContents();
                $parsedResponse = json_decode($body, true);
                if (isset($parsedResponse['data']['Message']))
                    $api_message = $parsedResponse['data']['Message'];
                elseif (isset($parsedResponse['mess']))
                    $api_message = $parsedResponse['mess'];
                else $api_message = "No Message Found";

                if (isset($parsedResponse['status'])) {
                    $status = $parsedResponse['status'];
                    if ($status == 'success' || $status == 'pending') {
                        $withdraw->status = "success";
                        $withdraw->save();
                        $user->transactions()->create([
                            'previous_amount' => $user->balance,
                            'amount' => $withdraw->amount,
                            'current_amount' => $withdraw->amount + $user->balance,
                            "type" => "withdraw",
                            "details" => "Withdraw ($withdraw->amount) Accepted"
                        ]);
                        $user->notify(new WithdrawRequestAcceptNotification($withdraw->amount, $user->fcm));
                        $success_request_count++;
                    } elseif ($status == 'failed') {
                        $failed_request_count++;
                    }
                } else {
                    $status_failed_request_count++;
                }
            } catch (\Exception $e) {
                $api_failed_request_count++;
            }
        }
        return back()->with('success', 'All Pending Requests has been accepted or rejected. Success: ' . $success_request_count . ', Failed: ' . $failed_request_count . ', Pending (will be success shortly): ' . $pending_request_count . ', Status Failed: ' . $status_failed_request_count . ', API Failed: ' . $api_failed_request_count);
    }

    private function acceptRequestApiIBRPay($id)
    {
        $withdraw = WithdrawHistory::where('id', $id)
            ->with('user', 'user.withdrawDetails')
            ->first();
        $user = $withdraw->user;

        $client = new Client();
        $url = 'https://ibrpay.com/api/PayoutLive.aspx';
        $data = [
            'APIID' => env('GATEWAY_IBR_PAY_APIID'),
            'Token' => env('GATEWAY_IBR_PAY_API_TOKEN'),
            'MethodName' => 'payout',
            'OrderID' => Str::random(12),
            'Name' => $user->withdrawDetails->account_holder_name,
            'Amount' => $withdraw->amount,
            'number' => $user->withdrawDetails->account_number,
            'ifsc' => $user->withdrawDetails->account_ifsc_code,
            'PaymentType' => 'personal',
            'CustomerMobileNo' => $user->phone,
        ];

        try {
            $response = $client->post($url, [
                'json' => $data
            ]);

            $statusCode = $response->getStatusCode();
            $body = $response->getBody()->getContents();
            $parsedResponse = json_decode($body, true);
            if (isset($parsedResponse['data']['Message']))
                $api_message = $parsedResponse['data']['Message'];
            elseif (isset($parsedResponse['mess']))
                $api_message = $parsedResponse['mess'];
            else $api_message = "No Message Found";

            if (isset($parsedResponse['status'])) {
                $status = $parsedResponse['status'];
                if ($status == 'success' || $status == 'pending') {
                    $withdraw->status = "success";
                    $withdraw->save();
                    $user->transactions()->create([
                        'previous_amount' => $user->balance,
                        'amount' => $withdraw->amount,
                        'current_amount' => $withdraw->amount + $user->balance,
                        "type" => "withdraw",
                        "details" => "Withdraw ($withdraw->amount) Accepted"
                    ]);
                    $user->notify(new WithdrawRequestAcceptNotification($withdraw->amount, $user->fcm));
                    $message = "Payment Request sent to API. Message from API: " . $api_message;
                    return $message;
                    return $message;
                } elseif ($status == 'failed') {
                    $message = "API Error, " . $api_message . ", Please Mannually Send Payment to user or Contact Support";
                    return $message;
                }
            } else {
                $message = "API Error, Invalid Status message, Status Code is: " . $parsedResponse['statusCode'];
                return $message;
            }
        } catch (\Exception $e) {
            $error = $e->getMessage();
            $message = "API Error: " . $error;
            return $message;
        }
    }

    private function acceptRequestApiUPIMoney($id)
    {
        $withdraw = WithdrawHistory::where('id', $id)
            ->with('user', 'user.withdrawDetails')
            ->first();
        $user = $withdraw->user;

        $client = new Client();
        $url = 'https://upimoney.co.in/api/payout/transaction';
        $data = [
            'token' => env('GATEWAY_UPI_MONEY_API_TOKEN'),
            'type' => 'bank',
            'name' => $user->withdrawDetails->account_holder_name,
            //as we don't have bank stored
            'bank' => "SBI",
            'account' => $user->withdrawDetails->account_number,
            'ifsc' => $user->withdrawDetails->account_ifsc_code,
            'mode' => 'IMPS',
            'mobile' => $user->phone,
            'email' => $user->phone . '@gmail.com',
            'address' => 'delhi',
            'apitxnid' => Str::random(12),
            'amount' => $withdraw->amount
        ];

        try {
            $response = $client->post($url, [
                'json' => $data
            ]);

            $body = $response->getBody()->getContents();
            $parsedResponse = json_decode($body, true);

            if (isset($parsedResponse['statuscode'])) {
                $statuscode = $parsedResponse['statuscode'];
                if (isset($parsedResponse['message']))
                    $api_message = $parsedResponse['message'];
                else $api_message = "No Message Found";

                if ($statuscode == 'ERR') {
                    $message = "API Error: " . $api_message;
                    return $message;
                } elseif ($statuscode == 'TXN') {
                    $status = $parsedResponse['status'];
                    if ($status == 'inprocess') {
                        $withdraw->status = "success";
                        $withdraw->save();
                        $user->transactions()->create([
                            'previous_amount' => $user->balance,
                            'amount' => $withdraw->amount,
                            'current_amount' => $withdraw->amount + $user->balance,
                            "type" => "withdraw",
                            "details" => "Withdraw ($withdraw->amount) Accepted"
                        ]);
                        $user->notify(new WithdrawRequestAcceptNotification($withdraw->amount, $user->fcm));
                        $message = "Payment Request sent to API. Message from API: " . $api_message;
                        return $message;
                    } else {
                        $message = "Unknown Status, Please Contact Support, Status Code is: " . $status;
                        return $message;
                    }
                } else {
                    $message = "API Error, Invalid Status message, Status Code is: " . $statuscode;
                    return $message;
                }
            } else {
                $message = "API Error, Invalid Status message, Status Code is: ";
                return $message;
            }
        } catch (\Exception $e) {
            $error = $e->getMessage();
            $message = "API Error: " . $error;
            return $message;
        }
    }

    public function acceptRequestApi($id)
    {
        if (!env('ENABLE_PAYOUT')) return back()->with('success', 'Payout is disabled');
        $appData = AppData::first();
        $payout_name = $appData->payment_method;

        //if already success then return error
        $withdraw = WithdrawHistory::findOrFail($id);
        if ($withdraw->status == "success") {
            return back()->with('error', 'Request has already been accepted');
        }

        if ($payout_name === 'ibr_pay') {
            $message = $this->acceptRequestApiIBRPay($id);
            return back()->with('success', $message);
        } elseif ($payout_name === 'upi_money') {
            $message = $this->acceptRequestApiUPIMoney($id);
            return back()->with('success', $message);
        } else {
            return back()->with('success', 'Payout API is not configured');
        }
    }

    public function acceptRequest($id)
    {
        $withdraw = WithdrawHistory::findOrFail($id);
        if ($withdraw->status == "success") {
            return back()->with('error', 'Request has already been accepted');
        }
        $withdraw->status = "success";
        $withdraw->save();
        $user = $withdraw->user;
        if ($user->role == 'agent') {
            $user->agentTransactions()->create([
                'previous_amount' => $user->balance,
                'amount' => $withdraw->amount,
                'current_amount' => $withdraw->amount + $user->balance,
                "type" => "withdraw",
                "details" => "Withdraw ($withdraw->amount) Accepted"
            ]);
        } else {
            $user->transactions()->create([
                'previous_amount' => $user->balance,
                'amount' => $withdraw->amount,
                'current_amount' => $withdraw->amount + $user->balance,
                "type" => "withdraw",
                "details" => "Withdraw ($withdraw->amount) Accepted"
            ]);
        }

        $user->notify(new WithdrawRequestAcceptNotification($withdraw->amount, $user->fcm));
        return back()->with('success', 'Request has been accepted');
    }

    public function rejectRequest($id)
    {
        Log::info('reject request called');
        $withdraw = WithdrawHistory::with("user")->findOrFail($id);
        if ($withdraw->status == "failed") {
            return back()->with('error', 'Request has already been rejected');
        }
        $withdraw->status = "failed";
        $withdraw->save();

        $previous_amount = $withdraw->user->balance;
        $current_amount = $previous_amount + $withdraw->amount;

        $user = $withdraw->user;
        $user->balance += $withdraw->amount;
        $user->save();
        if ($user->role == 'agent') {
            $user->agentTransactions()->create([
                'previous_amount' => $previous_amount,
                'amount' => $withdraw->amount,
                'current_amount' => $current_amount,
                "type" => "withdraw",
                "details" => "Withdraw ($withdraw->amount) Rejected"
            ]);
        } else {
            $user->transactions()->create([
                'previous_amount' => $previous_amount,
                'amount' => $withdraw->amount,
                'current_amount' => $current_amount,
                "type" => "withdraw",
                "details" => "Withdraw ($withdraw->amount) Rejected"
            ]);
        }

        $user->notify(new WithdrawRequestRejectNotification($withdraw->amount, $user->fcm));
        return back()->with('success', 'Request has been rejected');
    }
}
