<?php

namespace App\Http\Controllers\Dashboard;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Ramsey\Uuid\Type\Decimal;

class AgentController extends Controller
{
    public function destroy($id)
    {
        $user = User::where('id', $id)->first();
        if ($user === NULL) {
            return redirect()->route('agents.index')->with('error', 'Agent not found');
        }
        $user->withdrawDetails()->delete();
        $user->delete();
        return back()->with('success', 'Agent is deleted');
    }

    public function toogleBlock(Request $request)
    {
        $request->validate([
            'user_id' => 'required|numeric'
        ]);
        $user = User::findOrFail($request->user_id);
        $user->blocked = intval($request->blocked);
        $user->save();
    }

    public function index(Request $request)
    {
        if ($request->has('searchValue')) {
            $searchValue = $request->searchValue;
            $users = User::where('role', 'agent')
                ->where(function ($query) use ($searchValue) {
                    $query->where('name', 'LIKE', '%' . $searchValue . '%')
                        ->orWhere('phone', 'LIKE', '%' . $searchValue . '%');
                })->latest()->paginate(250);
            return view('dashboard.agents.index', compact('users', 'searchValue'));
        }

        $users = User::where('role', 'agent')->latest()->paginate(25);
        return view("dashboard.agents.index", [
            'users' => $users
        ]);
    }

    public function create()
    {
        return view('dashboard.agents.create');
    }

    public function getUserDetail($id)
    {
        $user = User::where('id', $id)->first();
        if ($user === NULL) {
            return redirect()->route('agents.index')->with('error', 'Agent not found');
        }

        return view('dashboard.agents.create', compact('user'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'phone' => [
                'required',
                'regex:/[0-9]{10}/',
                'digits:10',
                Rule::unique('users', 'phone')->ignore($request->id),
            ],
            'password' => [
                'nullable',
                'required_if:id,null',
                'confirmed',
                'min:4',
            ],
            'play_percentage' => 'required|numeric|min:0|max:100',
        ]);

        if ($request->id) {
            $user = User::where('id', $request->id)->first();
            if ($user === NULL) {
                return redirect()->route('agents.index')->with('error', 'Agent not found');
            }
            $user->name = $request->name;
            $user->phone = $request->phone;
            $user->password = Hash::make($request->password);
            $user->password_show = $request->password;
            $user->play_percentage = $request->play_percentage;
            $user->save();
        } else {
            do {
                $ownCode = 'A' . mt_rand(100000, 999999);
            } while (User::where('own_code', $ownCode)->exists());
            $user =   User::create([
                'name' => $request->name,
                'phone' => "$request->phone",
                'password' => Hash::make($request->password),
                'password_show' => $request->password,
                'user_id' => auth()->id(),
                'role' => 'agent',
                'own_code' => $ownCode,
                'confirmed' => 1,
                'fcm' => null,
                'blocked' => 0,
                'play_percentage' => $request->play_percentage,
            ]);

            //create withdrawDetails
            $user->withdrawDetails()->create();
        }

        return redirect()->route('agents.index')
            ->with('success', 'Agent successfully created');
    }
}
