<?php

namespace App\Http\Controllers\Agent;

use App\Models\AppData;
use Illuminate\Http\Request;
use App\Models\WithdrawHistory;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class WalletController extends Controller
{
    public function index(Request $request)
    {
        if ($request->has('searchValue')) {
            $searchValue = $request->searchValue;
            $withdrawHistories = WithdrawHistory::with('user')
                ->whereHas('user', function ($query) use ($searchValue) {
                    $query->where('name', 'LIKE', '%' . $searchValue . '%')
                        ->orWhere('phone', 'LIKE', '%' . $searchValue . '%')
                        ->where('role', 'agent')
                    ;
                })->where('user_id', auth()->user()->id)->latest()->paginate(250);
            return view('agent.wallet.index', compact('withdrawHistories', 'searchValue'));
        }
        $withdrawHistories  = WithdrawHistory::with('user')
            ->whereHas('user', function ($query) {
                $query->where('role', 'agent');
            })->where('user_id', auth()->user()->id)->latest()->paginate(25);
        return view('agent.wallet.index', compact('withdrawHistories'));
    }

    public function withdraw(Request $request)
    {
        Log::info($request->all());

        $request->validate([
            'amount' => 'required|numeric|min:1',
            'payment_method' => 'required|in:upi,bank',
            'upi_id' => 'required_if:payment_method,upi',
            'bank_name' => 'required_if:payment_method,bank',
            'account_holder_name' => 'required_if:payment_method,bank',
            'account_number' => 'required_if:payment_method,bank',
            'ifsc_code' => 'required_if:payment_method,bank',
        ]);

        if (env('AGENT_MIN_WITHDRAW', 1) > $request->amount) {
            return back()->withErrors('Minimum withdrawal amount is ' . env('AGENT_MIN_WITHDRAW'));
        }

        if (auth()->user()->balance < $request->amount) {
            return back()->withErrors('Insufficient balance');
        }

        $appData = AppData::first();

        if ($appData->withdrawal_fee === NULL || $appData->withdrawal_fee === 0) {
            $payabal_amount = $request->amount;
        } else {
            $payabal_amount = $request->amount - ($request->amount * $appData->withdrawal_fee / 100);
        }
        $withdraw_history = new WithdrawHistory();
        $withdraw_history->user_id = auth()->id();
        $withdraw_history->amount = $request->amount;
        $withdraw_history->payabal_amount = $payabal_amount;
        $withdraw_history->withdraw_mode = $request->payment_method;
        $withdraw_history->save();

        /** @var User $user */
        $user = Auth::user();
        $user->balance -= $request->amount;
        $user->save();

        if ($request->payment_method == 'upi') {
            $user->withdrawDetails()->updateOrCreate([
                'user_id' => $user->id
            ], [
                'upi_name' => $request->upi_name,
                'upi_id' => $request->upi_id,
            ]);
        } else {
            $user->withdrawDetails()->updateOrCreate([
                'user_id' => $user->id
            ], [
                'account_holder_name' => $request->account_holder_name,
                'account_number' => $request->account_number,
                'account_ifsc_code' => $request->ifsc_code,
            ]);
        }

        return back()->with('success', 'Withdraw request submitted successfully');
    }
}
